<?php

namespace App\Http\Controllers;

use App\Models\Branch;
use App\Models\MenuItem;
use App\Models\MenuPermission;
use App\Models\User;
use App\Models\UserPermission;
use Illuminate\Http\Request;

class AccessPermissionController extends Controller {
    public function index() {
        $users    = User::all();
        $branches = Branch::all();

        return view('permission.access-permission', compact('users', 'branches'));
    }

    public function store(Request $request) {
        $request->validate([
            'user_id'    => 'required|exists:users,id',
            'branch_id'  => 'required|exists:branches,id',
            'menu_items' => 'array|nullable',
        ]);

        $user   = User::find($request->user_id);
        $branch = Branch::find($request->branch_id);

        if ($user->role !== 'Operator') {
            $this->updateAllPermissions($user, $branch);
        } else {
            $this->updateOperatorPermissions($user, $branch, $request->menu_items);
        }

        return storeResponse();
    }

    private function updateAllPermissions($user, $branch) {
        $menuItems = MenuItem::all();

        foreach ($menuItems as $menuItem) {
            $this->syncPermission($user, $branch, $menuItem, true);
        }

    }

    private function updateOperatorPermissions($user, $branch, $menuItems) {
        $allMenuItems = MenuItem::all();

        foreach ($allMenuItems as $menuItem) {

            $canAccess = !is_null($menuItems) ? in_array($menuItem->id, $menuItems) : false;
            $this->syncPermission($user, $branch, $menuItem, $canAccess);

        }

    }

    private function syncPermission($user, $branch, $menuItem, $canAccess) {
        // dd($user, $branch, $menuItem, $canAccess);
        $userPermission = UserPermission::firstOrCreate([
            'user_id'   => $user->id,
            'branch_id' => $branch->id,
        ]);

        MenuPermission::updateOrCreate(
            [
                'user_permission_id' => $userPermission->id,
                'menu_item_id'       => $menuItem->id,
            ],
            ['can_access' => $canAccess]
        );
    }

    public function fetchMenuPermissionsHtml($userId, $branchId) {
        // Find the user and branch by ID
        $user   = User::findOrFail($userId);
        $branch = Branch::findOrFail($branchId);

        // Get all the menu items and check if the user has permissions for them
        $menuItems   = MenuItem::all();
        $permissions = $user->menuPermissions()->where('branch_id', $branchId)->get();

// dd($permissions);
        // Prepare HTML data for each menu item, including whether the user has permission for it
        $html = view('permission.menu-list', compact('menuItems', 'permissions'))->render();

        return response()->json(['html' => $html]);
    }

}
