<?php

namespace App\Http\Controllers;

use App\Jobs\SendAuthMail;
use App\Models\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller {
    public function register() {
        return view('auth.register');
    }

    public function registration(Request $request) {
        DB::beginTransaction();

        try {
            $validator = Validator::make($request->all(), [
                'name'     => 'required',
                'email'    => 'required|email|unique:users,email',
                'password' => 'required|min:3|confirmed',
            ]);

            if ($validator->fails()) {
                return back()->with('errors', $validator->messages()->all()[0])->withInput();
            }

            $ip      = request()->ip();
            $ip_data = json_decode(file_get_contents("http://ip-api.com/json/{$ip}"));

            $user = User::create([
                'name'        => $request->name,
                'email'       => $request->email,
                'password'    => bcrypt($request->password),
                'ip'          => $ip,
                'country'     => $ip_data->country ?? null,
                'city'        => $ip_data->city ?? null,
                'region_name' => $ip_data->regionName ?? null,
                'postal_code' => $ip_data->zip ?? null,
                'lat'         => $ip_data->lat ?? null,
                'lon'         => $ip_data->lon ?? null,
                'timezone'    => $ip_data->timezone ?? null,
            ]);

            $data            = [];
            $data['name']    = $user->name;
            $data['email']   = $user->email;
            $data['token']   = $request->_token;
            $data['subject'] = 'Account verification';
            $data['url']     = route('verifyAccount', ['token' => $request->_token, 'email' => $user->email]);
            $data['button']  = 'Verify Account';

            dispatch(new SendAuthMail($data));

            DB::table('password_reset_tokens')->insert([
                'type'       => 'account-verification',
                'token'      => $request->_token,
                'email'      => $request->email,
                'created_at' => now(),
            ]);
            DB::commit();

            return to_route('root')->withToastSuccess('Your account created successfully! We have sent a fresh account verification link to your email.');
        } catch (Exception $e) {
            DB::rollBack();

            return to_route('register')->withToastError('Something went wrong! Please try again later.');

        }

    }

    public function root() {
        return view('auth.login');
    }

    public function login(Request $request) {
        $validator = Validator::make($request->all(), [
            'email'    => 'required|min:2',
            'password' => 'required|min:3',
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        $credentials = $request->only('email', 'password');
        $data        = User::whereEmail($credentials['email'])->first();

        if (!$data) {
            return to_route('root')->withToastError('Invalid credentials.');
        } elseif (!password_verify($credentials['password'], $data->password)) {
            return to_route('root')->withToastError('Invalid credentials.');
        } elseif ($data->is_active != 1) {
            return to_route('root')->withToastError('Your account is inactive. Please contact admin.');
        } elseif ($data->email_verified_at == null) {
            session(['name' => $data->name]);

            return to_route('verify');
        } elseif (Auth::attempt($credentials)) {
            Auth::login($data);

            return to_route('dashboard.welcome');
        }

        return to_route('root')->withToastError('Invalid credentials.');
    }

    public function forgotPassword(Request $request) {

        return view('auth.forgot-password');
    }

    public function storeForgotPassword(Request $request) {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return back()->with('toast_error', $validator->messages()->all()[0])->withInput();
        }

        $user = User::whereEmail($request->email)->first();

        if (!$user) {
            return redirect()->back()->withToastError('This email is no longer with our records.');
        }

        $data            = [];
        $data['name']    = $user->name;
        $data['email']   = $user->email;
        $data['token']   = $request->_token;
        $data['subject'] = 'Password reset request';
        $data['url']     = route('resetPassword', [$request->_token, 'email' => $user->email]);
        $data['button']  = 'Reset Password';

        dispatch(new SendAuthMail($data));

        DB::table('password_reset_tokens')->insert([
            'type'       => 'reset-password',
            'token'      => $request->_token,
            'email'      => $request->email,
            'created_at' => now(),
        ]);

        return redirect()->back()->withToastSuccess('We have sent a fresh reset password link to your email.');
    }

    public function resetPassword(Request $request) {
        return view('auth.reset-password', ['request' => $request]);
    }

    public function storeResetPassword(Request $request) {
        $validator = Validator::make($request->all(), [
            'token'    => 'required',
            'email'    => 'required|email',
            'password' => 'required|min:3|confirmed',
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        $password = DB::table('password_reset_tokens')
            ->where('type', 'reset-password')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        if (!$password) {
            return redirect()->back()->with('Something went wrong! Invalid token or email.');
        }

        $user = User::whereEmail($request->email)->first();

        if ($user && $password) {
            $user->update(['password' => bcrypt($request->password)]);

            $password = DB::table('password_reset_tokens')
                ->where('type', 'reset-password')
                ->where('email', $request->email)
                ->delete();

            return to_route('root')->withToastSuccess('New password reset successfully. You can login now.');
        } else {
            return redirect()->back()->with('The email is no longer our records.');
        }

    }

    public function verify(Request $request) {
        $name = session('name');

        if (is_null($name)) {
            return to_route('root')->withToastError('Unauthorized access to verify page.');
        }

        return view('auth.verify');
    }

    public function storeVerifyAccount(Request $request) {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return back()->with('toast_error', $validator->messages()->all()[0])->withInput();
        }

        $user = User::whereEmail($request->email)->first();

        if (!$user) {
            return redirect()->back()->withToastError('This email is no longer with our records.');
        } elseif ($user->email_verified_at != null) {
            return redirect()->back()->withToastError('Your account is already verified. Please login.');
        }

        $data            = [];
        $data['name']    = $user->name;
        $data['email']   = $user->email;
        $data['token']   = $request->_token;
        $data['subject'] = 'Account verification request';
        $data['url']     = route('verifyAccount', ['token' => $request->_token, 'email' => $user->email]);
        $data['button']  = 'Verify Account';

        dispatch(new SendAuthMail($data));

        DB::table('password_reset_tokens')->insert([
            'type'       => 'account-verification',
            'token'      => $request->_token,
            'email'      => $request->email,
            'created_at' => now(),
        ]);

        session()->flush();

        return to_route('root')->withToastSuccess('We have sent a fresh account verification link to your email.');
    }

    public function verifyAccount(Request $request) {
        $validator = Validator::make($request->all(), [
            'token' => 'required',
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return to_route('root')->withToastError('Something went wrong! Please try again later or verify your account by requesting a new verification link.');
        }

        $password = DB::table('password_reset_tokens')
            ->where('type', 'account-verification')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        if (!$password) {
            return redirect()->back()->with('Something went wrong! Please try again later or verify your account by requesting a new verification link.');
        }

        $user = User::whereEmail($request->email)
            ->where('email_verified_at', null)
            ->first();

        if (!$user) {
            return redirect()->back()->with('Your account is already verified. Please login.');
        }

        if ($user && $password) {
            $user->update(['email_verified_at' => now()]);

            $password = DB::table('password_reset_tokens')
                ->where('type', 'account-verification')
                ->where('email', $request->email)
                ->delete();

            return to_route('root')->withToastSuccess('Your account has been verified successfully. You can login now.');
        } else {
            return redirect()->back()->with('The email is no longer with our records.');
        }

    }

    public function logout(Request $request) {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        session()->flush();

        return redirect()->route('root')->withToastSuccess('Successfully logged out.');
    }

}
