<?php

namespace App\Http\Controllers;

use App\Models\Journal;
use App\Models\JournalStatistic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class JournalStatisticController extends Controller {
    public function index(Request $request) {

        if ($request->ajax()) {
            $search = $request->get('search');

            $query = JournalStatistic::latest();

            if ($search) {
                $query->whereAny([
                    'volume',
                    'issue',
                ], 'like', "%{$search}%");
            }

            $data      = $query->paginate()->withQueryString();
            $rowsHtml  = view('journal-statistic.index-data-rows', compact('data'))->render();
            $linksHtml = $data->links('vendor.pagination.custom-pagination')->render();

            return response()->json([
                'rows'  => $rowsHtml,
                'links' => $linksHtml,
            ]);
        }

        $data = JournalStatistic::latest()->paginate();

        return view('journal-statistic.index', compact('data'));
    }

    public function create() {
        $journals = Journal::orderBy('name')->get();

        return view('journal-statistic.create', compact('journals'));
    }

    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
            'journal_id' => 'required|integer',
            'volume'     => [
                'required',
                'integer',
                // Unique Volume per Journal
                Rule::unique('journal_statistics')->where(fn($q) =>
                    $q->where('journal_id', $request->journal_id)
                ),
            ],
            'issue'      => [
                'required',
                'integer',
                // Unique Issue per Journal + Volume
                Rule::unique('journal_statistics')->where(fn($q) =>
                    $q->where('journal_id', $request->journal_id)
                        ->where('volume', $request->volume)
                ),
            ],
            'start_date' => 'required|date|after:today',
            'end_date'   => 'required|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        DB::beginTransaction();
        try {
            JournalStatistic::create($request->only(['journal_id', 'volume', 'issue', 'start_date', 'end_date']));

            DB::commit();

            return storeResponse();
        } catch (\Exception $e) {
            DB::rollBack();

            return errorResponse();
        }

    }

    public function show($id) {
        $data = JournalStatistic::where('id', $id)->first();

        if (!$data) {
            return errorResponse();
        }

        return view('journal-statistic.show', compact('data'));
    }

    public function edit($id) {
        $data = JournalStatistic::find($id);

        if (!$data) {
            return errorResponse();
        }

        $journals = Journal::orderBy('name')->get();

        return view('journal-statistic.edit', compact('data', 'journals'));
    }

    public function update(Request $request, $id) {
        $validator = Validator::make($request->all(), [
            'journal_id' => 'required|integer',
            'volume'     => [
                'required',
                'integer',
                // Unique Volume per Journal (Ignoring current ID)
                Rule::unique('journal_statistics')
                    ->where(fn($q) => $q->where('journal_id', $request->journal_id))
                    ->ignore($id),
            ],
            'issue'      => [
                'required',
                'integer',
                // Unique Issue per Journal + Volume (Ignoring current ID)
                Rule::unique('journal_statistics')
                    ->where(fn($q) =>
                        $q->where('journal_id', $request->journal_id)
                            ->where('volume', $request->volume)
                    )
                    ->ignore($id),
            ],
            'start_date' => 'required|date|after:today',
            'end_date'   => 'required|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        DB::beginTransaction();
        try {
            $data = JournalStatistic::find($id);

            if (!$data) {
                return errorResponse();
            }

            $payload = [
                'journal_id' => $request->journal_id,
                'volume'     => $request->volume,
                'issue'      => $request->issue,
                'start_date' => $request->start_date,
                'end_date'   => $request->end_date,
            ];

            $data->update($payload);

            return updateResponse('journal-statistics.index');
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function destroy($id) {
        $data = JournalStatistic::find($id);

        if (!$data) {
            return errorResponse();
        }

        $data->delete();

        return deleteResponse();
    }

}
