<?php

namespace App\Http\Controllers;

use App\Models\SiteDesignation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class SiteDesignationController extends Controller {
    public function index(Request $request) {

        if ($request->ajax()) {
            $search = $request->get('search');

            $query = SiteDesignation::query();

            if ($search) {
                $query->whereAny([
                    'name',
                    'note',
                ], 'like', "%{$search}%");
            }

            $data      = $query->paginate()->withQueryString();
            $rowsHtml  = view('site-designation.index-data-rows', compact('data'))->render();
            $linksHtml = $data->links('vendor.pagination.custom-pagination')->render();

            return response()->json([
                'rows'  => $rowsHtml,
                'links' => $linksHtml,
            ]);
        }

        $data = SiteDesignation::paginate();

        return view('site-designation.index', compact('data'));
    }

    public function create() {
        return view('site-designation.create');
    }

    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
            'name' => ['required', 'max:255', 'unique:site_designations,name'],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {

            SiteDesignation::create([
                'name' => $request->name,
                'note' => $request->note,
            ]);

            return storeResponse();
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function edit($id) {
        $data = SiteDesignation::find($id);

        if (!$data) {
            return errorResponse();
        }

        return view('site-designation.edit', compact('data'));
    }

    public function update(Request $request, $id) {
        $validator = Validator::make($request->all(), [
            'name' => ['required', 'string', 'max:255', Rule::unique('site_designations', 'name')->ignore($id)],
        ]);

        if ($validator->fails()) {
            return back()->with('errors', $validator->messages()->all()[0])->withInput();
        }

        try {
            $data = SiteDesignation::find($id);

            if (!$data) {
                return errorResponse();
            }

            $payload = [
                'name' => $request->name,
                'note' => $request->note,
            ];

            $data->update($payload);

            return updateResponse('site-designations.index');
        } catch (\Exception $e) {
            return errorResponse();
        }

    }

    public function destroy($id) {
        $data = SiteDesignation::find($id);

        if (!$data) {
            return errorResponse();
        }

        $data->delete();

        return deleteResponse();
    }

}
