<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class Journal extends Model {
    protected $guarded = [];

    public function journalCategories() {
        return $this->hasMany(JournalCategory::class, 'journal_id');
    }

    public function createdBy() {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updatedBy() {
        return $this->belongsTo(User::class, 'updated_by');
    }

    protected static function boot() {
        parent::boot();

        static::creating(function ($model) {
            $model->created_by = Auth::id();
        });

        static::created(function ($model) {
            $journal = self::with('journalCategories')->where('id', $model->id)->first();

            Cache::forever(self::cacheKey($journal->slug), $journal);
            Cache::forget(self::listCacheKey());
        });

        static::updating(function ($model) {
            $model->updated_by = Auth::id();
        });

        static::updated(function ($model) {
            $journal = self::with('journalCategories')->where('id', $model->id)->first();

            Cache::forever(self::cacheKey($journal->slug), $journal);
            Cache::forget(self::listCacheKey());
        });

        static::deleted(function ($model) {
            Cache::forget(self::cacheKey($model->slug));
            Cache::forget(self::listCacheKey());
        });
    }

    public static function cacheKey($slug): string {
        return "journal_{$slug}";
    }

    public static function listCacheKey(): string {
        return "journal_list";
    }

    public function assignments() {
        return $this->hasMany(JournalUserAssignment::class);
    }

    public function editors()
    {
        return $this->belongsToMany(
            User::class,
            'journal_user_assignments',
            'journal_id',      // FK on pivot → Journal
            'assignable_id'    // FK on pivot → User
        )
        ->wherePivot('role', 'Editor')
        ->wherePivot('assignable_type', User::class);
    }

    public function reviewers()
    {
        return $this->belongsToMany(
            User::class,
            'journal_user_assignments',
            'journal_id',
            'assignable_id'
        )
        ->wherePivot('role', 'Reviewer')
        ->wherePivot('assignable_type', User::class);
    }
}

/*

Single journal rendering example:
$journal = Cache::rememberForever(
Journal::cacheKey($id),
fn () => Journal::findOrFail($id)
);

list of journals rendering example:
$journals = Cache::rememberForever(
Journal::listCacheKey(),
fn () => Journal::orderBy('id', 'desc')->get()
);

 */