<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable {
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;
    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $guarded = [];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array {
        return [
            'email_verified_at' => 'datetime',
            'dob'               => 'date',
            'password'          => 'hashed',
        ];
    }

    public function siteDesignation() {
        return $this->belongsTo(SiteDesignation::class, 'site_designation_id');
    }

    public function userPermissions() {
        return $this->hasMany(UserPermission::class);
    }

    public function menuPermissions() {
        return $this->hasManyThrough(MenuPermission::class, UserPermission::class);
    }

    public function canAccessMenu(int $branchId, int | string | MenuItem $menu): bool {
        $menuItemId = $menu instanceof MenuItem
        ? $menu->id
        : (is_numeric($menu)
            ? (int) $menu
            : MenuItem::where('slug', $menu)->value('id'));

        if (!$menuItemId) {
            return false;
        }

        return $this->menuPermissions()
            ->where('menu_item_id', $menuItemId)
            ->where('can_access', true)
            ->whereHas('userPermission', fn($q) => $q->where('branch_id', $branchId))
            ->exists();
    }

    public function menuItemsForBranch(int $branchId) {
        return MenuItem::query()
            ->join('menu_permissions as mp', 'mp.menu_item_id', '=', 'menu_items.id')
            ->join('user_permissions as up', 'up.id', '=', 'mp.user_permission_id')
            ->where('up.user_id', $this->id)
            ->where('up.branch_id', $branchId)
            ->where('mp.can_access', true)
            ->select('menu_items.*')
            ->distinct();
    }

    public function journalAssignments() {
        return $this->morphMany(
            JournalUserAssignment::class,
            'assignable'
        );
    }

    public function reviewingJournals() {
        return $this->morphToMany(
            Journal::class,
            'assignable',
            'journal_user_assignments'
        )->wherePivot('role', 'reviewer');
    }

    public function editingJournals() {
        return $this->morphToMany(
            Journal::class,
            'assignable',
            'journal_user_assignments'
        )->wherePivot('role', 'editor');
    }

}
